# Load necessary libraries
library(Synth)
library(tidyverse)


## read in the example data
sugar_tax <- read.csv("example_data.csv")

summary(sugar_tax)

# Prepare data for Synth package
dataprep_out <- dataprep(
  foo = sugar_tax,
  predictors = c("gdp_per_capita", "mean_age", "urbanisation_rate", "mean_daily_calorie_intake"),
  predictors.op = "mean",
  dependent = "obesity_rate",
  unit.variable = "ID",
  time.variable = "year",
  treatment.identifier = 1,
  controls.identifier = c(2:4),
  time.predictors.prior = 2010:2019, # Pre-intervention period
  time.optimize.ssr = 2010:2019,
  unit.names.variable = "country",
  time.plot = 2010:2024
)



# Run synthetic control analysis
synth_out <- synth(dataprep_out)


## there are two ways to summarize the results
## we can either access the output from synth.out directly
round(synth_out$solution.w,4)
# contains the unit weights or
synth_out$solution.v 
## contains the predictor weights. 


## the output from synth opt 
## can be flexibly combined with 
## the output from dataprep to 
## compute other quantities of interest
## for example, the period by period 
## discrepancies between the 
## treated unit and its synthetic control unit
## can be computed by typing
gaps<- dataprep_out$Y1plot-(
  dataprep_out$Y0plot%*%synth_out$solution.w
) ; gaps

A_obesity <- as.numeric(dataprep_out$Y1plot)
A_obesity_synth <- as.numeric(dataprep_out$Y0plot%*%synth_out$solution.w)
Gap <- as.numeric (gaps)


A_out <- data.frame(Year = 2010:2024, A_obesity, A_obesity_synth, Gap)
A_out
write.csv(A_out,
          file = "synth_compare.csv",
          row.names = FALSE)

## also there are three convenience functions to summarize results.
## to get summary tables for all information 
## (V and W weights plus balance btw. 
## treated and synthetic control) use the 
## synth.tab() command
synth_tables <- synth.tab(
  dataprep.res = dataprep_out,
  synth.res = synth_out)
print(synth_tables)


# Plot results
path.plot(synth_out, dataprep_out, Ylab = "Obesity Rate", Xlab = "Year", 
          Legend = c("Country A", "Synthetic Control"), Legend.position = "topright")
abline(v = 2020, col = "red", lty = 2) # Policy intervention year
title("Synthetic Control Analysis of Sugar Tax on Obesity Rate")

# Print weights for synthetic control
print(synth_out$solution.w)

## plot the gaps (treated - synthetic)
gaps.plot(dataprep.res = dataprep_out,synth.res = synth_out)



